/*
 * The akquinet maven-latex-plugin project
 *
 * Copyright (c) 2011 by akquinet tech@spree GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.simuline.m2latex.core;
// is AbstractLatexMojo but not public 
import eu.simuline.m2latex.mojo.CfgLatexMojo;// for javadoc only 

import java.io.File;

import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.maven.plugins.annotations.Parameter;

/**
 * The settings for a maven plugin and for an ant task. 
 * These are the elements of the maven pom in element <code>settings</code> 
 * and accordingly for the ant build file. 
 */
public class Settings {

    // static initializer 

    /**
     * On unix <code>src/site/tex</code>, 
     * on other operating systems accordingly. 
     */
    final static String SST;

    static {
	String fs = System.getProperty("file.separator");
	SST = "src" + fs + "site" + fs + "tex";
    }

    // readonly parameters 

    /**
     * The base directory of this maven project. 
     *
     * @see CfgLatexMojo#baseDirectory
     */
    private File baseDirectory;

    /**
     * The target directory of this maven project. 
     * By default this is <code>{@link #baseDirectory}/target</code> 
     * on Unix systems. 
     *
     * @see CfgLatexMojo#targetDirectory
     */
    private File targetDirectory;

    /**
     * The target site directory of this maven project. 
     * By default this is <code>{@link #targetDirectory}/site</code> 
     * on Unix systems. 
     *
     * @see CfgLatexMojo#targetSiteDirectory
     */
    private File targetSiteDirectory;

    // read/write parameters and related. 
    // If a parameter represents a relative path, this is a string 
    // and there is an according field of type File. 

    /**
     * The latex source directory as a string 
     * relative to {@link #baseDirectory}, 
     * containing {@link #texSrcProcDirectory}. 
     * This directory determines also the subdirectory of 
     * {@link #outputDirectory} to lay down the generated artifacts. 
     * The according file is given by {@link #texSrcDirectoryFile}. 
     * The default value is {@link #SST}. 
     */
    @Parameter(name = "texSrcDirectory", 
	       defaultValue = "src${file.separator}site${file.separator}tex")
    private String texSrcDirectory = SST;

    /**
     * File for {@link #texSrcDirectory} based on {@link #baseDirectory}. 
     */
    private File texSrcDirectoryFile = new File(this.baseDirectory, 
						this.texSrcDirectory);

    /**
     * The latex source processing directory as a string 
     * relative to {@link #texSrcDirectory}
     * containing all tex main documents 
     * and the graphic files to be processed 
     * and also to be cleaned. 
     * Whether this is done recursively in subfolders 
     * is specified by {@link #readTexSrcProcDirRec}. 
     * The according file is given by {@link #texSrcProcDirectoryFile}. 
     * The default value is <code>.</code>, 
     * i.e. the latex souce processing directory is the latex source directory. 
     */
    @Parameter(name = "texSrcProcDirectory", defaultValue = ".")
    private String texSrcProcDirectory = ".";

    /**
     * File for {@link #texSrcProcDirectory} based on {@link #texSrcDirectory}. 
     */
    private File texSrcProcDirectoryFile = new File(this.texSrcDirectoryFile,
						    this.texSrcProcDirectory);

    /**
     * Whether the tex source directory {@link #texSrcProcDirectory} 
     * shall be read recursively, 
     * i.e. including the subdirectories recursively. 
     * This is set to <code>false</code> only during information development. 
     * The default value is <code>true</code>. 
     */
    // FIXME: maybe in the long run: only latex main files. 
    @Parameter(name = "readTexSrcProcDirRec", defaultValue = "true")
    private boolean readTexSrcProcDirRec = true;

    /**
     * The artifacts generated by {@link #latex2pdfCommand} 
     * will be copied to this folder 
     * which is given relative to {@link #targetSiteDirectory}. 
     * The default value is <code>.</code>. 
     * The according file is given by {@link #outputDirectoryFile}. 
     *
     * @see #texSrcDirectory
     */
    @Parameter(name = "outputDirectory", defaultValue = ".")
    private String outputDirectory = ".";

    /**
     * File for {@link #outputDirectory} based on {@link #targetSiteDirectory}. 
     */
    private File outputDirectoryFile = new File(this.targetSiteDirectory, 
						this.outputDirectory);

    /**
     * A comma separated list of targets 
     * returned as a set by {@link #getTargetSet()}. 
     * For allowed values see {@link Target}. 
     * The default value is <code>pdf, html</code>. 
     */
    @Parameter(name = "targets", defaultValue = "pdf, html")
    private String targets = "pdf, html";


    /**
     * The pattern to be applied to the beginning of the contents of tex-files 
     * which identifies a latex main file. 
     * The default value is choosen to match quite exactly 
     * the latex main files. 
     * Here we assume that the latex main file should contain 
     * the declaration `\documentclass' 
     * or the old fashioned `documentstyle' 
     * preceeded by a few constucts. 
     * Strictly speaking, this is not necessary. 
     * For a more thorough discussion, 
     * and for an alternative approach, consult the manual. 
     * <p>
     * The default value is choosen to match quite exactly 
     * the latex main files, no more no less. 
     * Since the pattern is chosen 
     * according to documentation collected from the internet, 
     * one can never be sure whether the pattern is perfect. 
     * <p>
     * If the current default value is not appropriate, 
     * please overwrite it in the configuration 
     * and notify the developer of this plugin of the deficiency. 
     */
    // FIXME: not only on this pattern: 
    // Matching is linewise which is inappropriate. 
    // pattern is to be applied to the start of the tex-file 
    @Parameter(name = "patternLatexMainFile")
    private String patternLatexMainFile = 
	"\\A(\\\\RequirePackage\\s*" +           // RequirePackage 
	/**/"(\\[(\\s|\\w|,)*\\])?\\s*" +    // [options]
	/**/"\\{\\w+\\}\\s*(\\[(\\d|\\.)+\\])?|" + // {name}version
	"%.*$|" + // comments 
	"\\\\PassOptionsToPackage\\s*" +     // PassOptionsToPackage
	/**/"\\{\\w+\\}\\s*" +    // {options}
	/**/"\\{\\w+\\}|" +       // {packagename}
	"%.*$|" + // comments 
	"\\\\input\\{[^{}]*\\}|" + // inputs 
	"\\s)*" + // spaces FIXME: quicker were \s* but BUG IN MATCHER 
	"\\\\(documentstyle|documentclass)";


    // texPath, commands and arguments 

    /**
     * Path to the TeX scripts or <code>null</code>. 
     * In the latter case, the scripts must be on the system path. 
     * Note that in the pom, <code>&lt;texPath/&gt;</code> 
     * and even <code>&lt;texPath&gt;    &lt;/texPath&gt;</code> 
     * represent the <code>null</code>-File. 
     * The default value is <code>null</code>. 
     */
    @Parameter(name = "texPath")//, defaultValue = null
    private File texPath = null;

    /**
     * Clean up the working directory in the end? 
     * May be used for debugging when setting to <code>false</code>. 
     * The default value is <code>true</code>. 
     */
    @Parameter(name = "cleanUp", defaultValue = "true")
    private boolean cleanUp;

    /**
     * This pattern is applied to file names 
     * and matching shall accept all the files 
     * which were created from a latex main file <code>xxx.tex</code>. 
     * It is neither applied to directories 
     * nor to <code>xxx.tex</code> itself. 
     * It shall not comprise neither graphic files to be processed 
     * nor files created from those graphic files. 
     * <p>
     * This pattern is applied 
     * in the course of processing graphic files 
     * to decide which graphic files should be processed 
     * (those rejected by this pattern) 
     * and to log warnings if there is a risk, 
     * that graphic files to be processed 
     * are skipped or that processing a latex main file overwrites 
     * the result of graphic preprocessing. 
     * <p>
     * When clearing the tex source directory {@link #texSrcProcDirectory}, 
     * i.e. all generated files should be removed, 
     * first those created from latex main files. 
     * As an approximation, 
     * those are removed which match this pattern. 
     * <p>
     * The sequence <code>T$T</code> 
     * is replaced by the prefix <code>xxx</code>. 
     * The sequence <code>T$T</code> must always be replaced: 
     * The symbol <code>$</code> occurs as end-sign as <code>)$</code> 
     * or as literal symbol as <code>\$</code>. 
     * Thus <code>T$T</code> is no regular occurrence 
     * and must always be replaced with <code>xxx</code>. 
     * <p>
     * Spaces and newlines are removed 
     * from that pattern before matching. 
     * <p>
     * This pattern may never be ensured to be complete, 
     * because any package 
     * may create files with names matching its own patterns 
     * and so any new package may break completeness. 
     * <p>	 
     * If the current default value is not appropriate, 
     * please overwrite it in the configuration 
     * and notify the developer of this plugin of the deficiency. 
     * The default value is given below. 
     */
    @Parameter(name = "patternCreatedFromLatexMain")
    private String patternCreatedFromLatexMain = 
	// besides T$T.xxx, with xxx not containing ., 
	// we allow T$T.synctex.gz and T$T.out.ps 
	"^(T$T(\\.([^.]*|synctex\\.gz(\\(busy\\))?|out\\.ps)|" + 
	// tex4ht creates files T$Tyy.(x)htm(l)... 
	"(-|ch|se|su|ap|li)?\\d+\\.x?html?|" + 
	// ... and T$Tddx.(x)bb, T$Tddx.png and T$T-dd.svg... 
	"\\d+x\\.x?bb|" + 
	"\\d+x?\\.png|" + 
	"-\\d+\\.svg|" + 
	// by (splitidx and) splitindex 
	"-.+\\.(idx|ind|ilg))|" + 
	// ... and xxT$T.eps... 
	"zzT$T\\.e?ps|" + 
	// ... and scripts cmsy....png 
	"(cmsy)\\d+(-c)?-\\d+c?\\.png|" + 
	// The following occurs sporadic 
	"(pdf)?latex\\d+\\.fls)$";


    // parameters for graphics preprocessing 


    /**
     * The fig2dev command for conversion of fig-files 
     * into various formats. 
     * Currently only pdf combined with pdf_t is supported. 
     * Note that preprocessing one fig-file 
     * requires two invocations of {@link #fig2devCommand}, 
     * one for each part. 
     * The default value is <code>fig2dev</code>. 
     *
     * @see #fig2devGenOptions
     * @see #fig2devPtxOptions
     * @see #fig2devPdfEpsOptions
     */
    @Parameter(name = "fig2devCommand", defaultValue = "fig2dev")
    private String fig2devCommand = "fig2dev";

    /**
     * The options for the command {@link #fig2devCommand} 
     * common to both output languages. 
     * For the options specific for the two output langugages 
     * <code>pdftex</code> and <code>pdftex_t</code>, 
     * see {@link #fig2devPtxOptions} and {@link #fig2devPdfEpsOptions}, 
     * respectively. 
     * The default value is the empty string. 
     * <p>
     * Possible are the following options: 
     * <ul>
     * <li><code>-D +/-rangelist</code> 
     * Export layers selectively (<code>+</code>) 
     * or exclude layers from export (<code>-</code>). 
     * E.g. -D +10,40,55:70,80  means  keep 
     * only layers 10, 40, 55 through 70, and 80.
     * <li><code>-j</code> 
     * i18n (internationalization feature)
     * <li><code>-m mag</code> 
     * Set the magnification at which the figure is rendered 
     * to <code>mag</code>.
     * The default is <code>1.0</code>. 
     * This is not usable within latex; not even <code>1.0</code>. 
     * <li><code>-s fsize</code> 
     * Set the default font size (in points) 
     * for text objects to <code>fsize</code>.
     * Refers to the latex-fonts only. 
     * <li><code>-b width</code> 
     * specify width of blank border around figure (1/72 inch). 
     * </ul>
     * Except for the option <code>-j</code>, 
     * all these options take parameters 
     * and it may make sense to use them with different parameters 
     * for the two output languages. 
     * In this case include them in 
     * {@link #fig2devPtxOptions} and in {@link #fig2devPdfEpsOptions}. 
     */
    @Parameter(name = "fig2devGenOptions", defaultValue = "")
    private String fig2devGenOptions = "";

    /**
     * The options for the command {@link #fig2devCommand} 
     * specific for the output languages <code>pdftex_t</code> 
     * and <code>pstex_t</code> which are the same. 
     * Note that in addition to these options, 
     * the option <code>-L pdftex_t</code> specifies the language, 
     * {@link #fig2devGenOptions} specifies the options 
     * common for the two output langugages 
     * <code>pdftex</code> and <code>pdftex_t</code> 
     * and <code>-p xxx</code> specifies the full path 
     * of the pdf/eps-file to be included without extension. 
     * <p>
     * The default value for this option is the empty string. 
     * <p>
     * Possible options are the following: 
     * (These seem to work for tex only 
     * although according to documentation for all languages. )
     * <ul>
     * <li> options specified for {@link #fig2devGenOptions} 
     * <li> <code>-E num</code>
     * Set encoding for latex text translation 
     * (0 no translation, 1 ISO-8859-1, 2 ISO-8859-2), 
     * others allowed also, effect not clear. 
     * <li> <code>-F</code>  
     * don't set font family/series/shape, 
     * so you can set it from latex. 
     * <li> <code>-v</code>
     * Verbose mode.
     * </ul>
     */
    // Note that several options do not make sense as global options, 
    // better as individual options. 
    // Maybe it makes sense, to include those options 
    // in the fig-file and use a wrapper around fig2dev 
    // instead of fig2dev itself, 
    // which invokes fig2dev with the according options. 
    // Problem is that xfig does not support this. 
    @Parameter(name = "fig2devPtxOptions", defaultValue = "")
    private String fig2devPtxOptions = "";

    /**
     * The options for the command {@link #fig2devCommand} 
     * specific for the output language <code>pdftex</code>. 
     * Note that in addition to these options, 
     * the option <code>-L pdftex</code> specifies the language and 
     * {@link #fig2devGenOptions} specifies the options 
     * common for the two output langugages 
     * <code>pdftex</code> and <code>pdftex_t</code>. 
     * The default value for this option is the empty string. 
     * <p>
     * Possible options are the following: 
     * (These seem to work specifically for pdf 
     * although according to documentation for all languages. )
     * <ul>
     * <li> options specified for {@link #fig2devGenOptions} 
     * <li> <code>-G minor[:major][unit]</code>
     * Draws a grid on the page.  
     *    e.g. "-G .25:1cm" draws a thin line every .25 cm 
     *    and a thicker line every 1 cm. 
     *    Default unit is in.  
     *    Allowable units are: 
     *    i, in, inch, f, ft, feet, c, cm, mm, and m. 
     * <li> <code>-A</code>
     * Add an ASCII (EPSI) preview.
     * <li> <code>-c</code>
     * centers the figure on the page.  (default)
     * seems not to have an effect...
     * <li> <code>-e</code>
     * puts the  figure against the edge (not centered) of the page. 
     * seems not to have an effect...
     * <li> <code>-F</code>
     * Use correct font sizes (points) instead of the traditional  size
     * <li> <code>-g color</code>
     * Use color for the background. 
     * FIXME: Not clear how to specify the color. 
     * <li> <code>-N</code>
     * Convert all colors to grayscale. (not available for latex fonts)
     * <li> <code>-n name</code>
     * Set  the /Title(xxx) of the PostScript output to <code>name</code>. 
     * without it is just the filename <code>xxx.fig</code>. 
     */
    // Note that several options do not make sense as global options, 
    // better as individual options. 
    // Maybe it makes sense, to include those options 
    // in the fig-file and use a wrapper around fig2dev 
    // instead of fig2dev itself, 
    // which invokes fig2dev with the according options. 
    // Problem is that xfig does not support this. 
    @Parameter(name = "fig2devPdfEpsOptions", defaultValue = "")
    private String fig2devPdfEpsOptions = "";

    /**
     * The command for conversion of gnuplot-files 
     * into various formats. 
     * Currently only pdf (graphics) 
     * combined with pdf_t (latex-texts) is supported. 
     * The default value is <code>gnuplot</code>. 
     */
    @Parameter(name = "gnuplotCommand", defaultValue = "gnuplot")
    private String gnuplotCommand = "gnuplot";

    /**
     * The options specific for {@link #gnuplotCommand}'s 
     * output terminal <code>cairolatex</code>, 
     * used for mixed latex/pdf-creation. 
     * <p>
     * Possible values are: 
     * <ul>
     * <li><code>{standalone | input}</code>
     * <li><code>{blacktext | colortext | colourtext}</code>
     * Specifies whether for text colors are taken into account or not. 
     * For all but text see separate options. 
     * <li><code>{header <header> | noheader}</code>
     * <li><code>{mono|color}</code>
     * Specifies whether colors are taken into account or not. 
     * Refers to all but text (for text see separate options)
     * <li><code>{{no}transparent} {{no}crop} {background <rgbcolor>}</code>
     * <li><code>{font <font>}</code>
     * <li><code>{fontscale <scale>}</code>
     * <li><code>{linewidth <lw>} {rounded|butt|square} {dashlength <dl>}</code>
     * <li><code>{size <XX>{unit},<YY>{unit}}</code>
     * The size of this picture. 
     * This is not usable, because it imposes deformation. 
     * Default unit is inch (<code>in</code>). 
     * </ul>
     * Note that the option <code>pdf|eps</code> 
     * of the terminal <code>cairolatex</code> is not available, 
     * because it is set internally. 
     * The default option string is empty. 
     */
    @Parameter(name = "gnuplotOptions", defaultValue = "")
    private String gnuplotOptions = "";

    /**
     * The command for conversion of gnuplot-files 
     * into metapost's postscript. 
     * The default value is <code>mpost</code>. 
     */
    @Parameter(name = "metapostCommand", defaultValue = "mpost")
    private String metapostCommand = "mpost";

    /**
     * The options for the command {@link #metapostCommand}. 
     * Leading and trailing blanks are ignored. 
     * A sequence of at least one blank separate the proper options. 
     * The default value comprises the following options: 
     * <ul>
     * <li><code>-interaction=nonstopmode</code> 
     * prevents metapost from stopping at the first error. 
     * <li><code>-recorder</code> 
     * makes metapost create an fls-file specifying all inputted files. 
     * <li><code>-s prologues=2</code> 
     * makes metapost create a postscript file 
     * which is viewable by ghostscript viewer.
     * </ul>
     * 
     * -debug creates intermediate files mp3mnuvD.dvi and mp3mnuvD.tex 
     * No info available about the details. 
     */
    @Parameter(name = "metapostOptions", 
	       defaultValue = "-interaction=nonstopmode -recorder " + 
	       "-s prologues=2")
    private String metapostOptions = 
	"-interaction=nonstopmode -recorder -s prologues=2";

    /**
     * The command for conversion of svg-files 
     * into a mixed format FIXME, synchronize with fig2devCommand. 
     * The default value is <code>inkscape</code>. 
     */
    @Parameter(name = "svg2devCommand", defaultValue = "inkscape")
    private String svg2devCommand = "inkscape";

    /**
     * The options for the command {@link #svg2devCommand} 
     * for exporting svg-figures into latex compatible files. 
     * <p>
     * The following options are mandatory: 
     * <ul>
     * <li><code>-z</code> or <code>--without-gui</code> 
     * process files from console. 
     * <li><code>-D</code> or <code>--export-area-drawing</code> 
     * Export the drawing (not the page)
     * <li><code>--export-latex</code> 
     * Export PDF/PS/EPS without text. 
     * Besides the PDF/PS/EPS, a LaTeX file is exported,
     * putting the text on top of the PDF/PS/EPS file. 
     * Include the result in LaTeX like: \input{latexfile.tex}. 
     * Note that the latter option is necessary, 
     * to create the expected files. 
     * It is also conceivable to export text as pdf/eps 
     * </ul>
     * <p>
     * The following options are prohibited, 
     * because they are automatically added by the software: 
     * <ul>
     * <li><code>-A=FILENAME</code> or <code>--export-pdf=FILENAME</code> 
     * Export document to a  PDF file. 
     * <li><code>-E=FILENAME</code> or <code>--export-eps=FILENAME</code> 
     * Export document to an EPS file. 
     * <ul>
     *
     * The default value is the minimal value, 
     * <code>-z -D --export-latex</code>. 
     */
   @Parameter(name = "svg2devOptions", defaultValue = "-z -D --export-latex")
   private String svg2devOptions = "-z -D --export-latex";

    /**
     * The command to create bounding box information 
     * from jpg-files and from png-files. 
     * This is run twice: 
     * once with parameter <code>-m</code> 
     * to create <code>.bb</code>-files for driver <code>dvipdfm</code> and 
     * once with parameter <code>-x</code> 
     * to create <code>.xbb</code>-files for driver <code>dvipdfmx</code>. 
     * The default value is <code>ebb</code>. 
     */
    @Parameter(name = "ebbCommand", defaultValue = "ebb")
    private String ebbCommand = "ebb";

    /**
     * The options for the command {@link #ebbCommand} 
     * except <code>-m</code> and <code>-x</code> 
     * which are added automatically. 
     * The default value is <code>-v</code>. 
     */
    // without -x and -m 
    @Parameter(name = "ebbOptions", defaultValue = "-v")
    private String ebbOptions  = "-v";

    // parameters for latex2pdf-conversion 

    /**
     * The LaTeX command to create a pdf-file or a dvi-file. 
     * FIXME: documentation must be updated. 
     * Possible values are e.g. 
     * <code>pdflatex</code>, <code>lualatex</code> and <code>xelatex</code>. 
     * The default value (for which this software is also tested) 
     * is <code>pdflatex</code>. 
     */
    @Parameter(name = "latex2pdfCommand", defaultValue = "pdflatex")
    private String latex2pdfCommand = "pdflatex";

    /**
     * The options for the command {@link #latex2pdfCommand}. 
     * Leading and trailing blanks are ignored. 
     * The setter method {@link #setLatex2pdfOptions(String)} ensures, 
     * that exactly one blank separate the proper options. 
     * <p>
     * The default value comprises the following options: 
     * <ul>
     * <li><code>-interaction=nonstopmode</code> 
     * prevents latex from stopping at the first error. 
     * <li><code>-synctex=1</code> 
     * makes latex create a pdf file 
     * which synchronizes with an editor supporting synchtex. 
     * <li><code>-src-specials</code> 
     * includes source specials into the output. dvi only? 
     * <li><code>-recorder</code> 
     * makes latex create an fls-file specifying all inputted files. 
     * <li><code>-shell-escape</code> 
     * allows to use write18-mechanism for shell commands (why needed?)
     * </ul>
     */
    // useful also: -file-line-error
    @Parameter(name = "latex2pdfOptions", 
	       defaultValue = "-interaction=nonstopmode " + // 
	"-synctex=1 " + 
	"-src-specials " + 
	"-recorder " + 
	"-shell-escape")
    private String latex2pdfOptions = 
	"-interaction=nonstopmode " + // 
	"-synctex=1 " + 
	"-src-specials " + 
	"-recorder " + 
	"-shell-escape";

    /**
     * The pattern is applied linewise to the log-file 
     * and matching indicates an error 
     * emitted by the command {@link #latex2pdfCommand}. 
     * <p>
     * The default value is choosen to match quite exactly 
     * the latex errors in the log file, no more no less. 
     * Since no official documentation was found, 
     * the default pattern may be incomplete. 
     * In fact it presupposes, that $latex2pdfOptions 
     * does not contain `<code>-file-line-error-style</code>'.   
     * <p>
     * If the current default value is not appropriate, 
     * please overwrite it in the configuration 
     * and notify the developer of this plugin of the deficiency. 
     * The default value is `<code>(^! )</code>' (note the space). 
     */
    // FIXME: Problem with line error style 
    @Parameter(name = "patternErrLatex", defaultValue = "(^! )")
    private String patternErrLatex = "(^! )";

    /**
     * The pattern is applied linewise to the log-file 
     * and matching indicates a warning 
     * emitted by the command {@link #latex2pdfCommand}, 
     * disragarding warnings on bad boxes 
     * provided {@link #debugWarnings} is set. 
     * <p>
     * This pattern may never be ensured to be complete, 
     * because any package may indicate a warning 
     * with its own pattern any new package may break completeness. 
     * Nevertheless, the default value aims completeness 
     * while be restrictive enough 
     * not to indicate a warning where none was emitted. 
     * <p>
     * If the current default value is not appropriate, 
     * please overwrite it in the configuration 
     * and notify the developer of this plugin of the deficiency. 
     * The default value is given below. 
     *
     * @see #debugBadBoxes
     */
     @Parameter(name = "patternWarnLatex", 
		defaultValue = 
		"^(LaTeX Warning: |" +
		"LaTeX Font Warning: |" + 
		"(Package|Class) .+ Warning: |" + 
		// pdftex warning (ext4): destination with the same identifier
		// pdfTeX warning (dest): ... has been referenced ...
		// pdfTeX warning: pdflatex (file pdftex.map): cannot open font map file 
		// pdfTeX warning: Found pdf version 1.5, allowed maximum 1.4 
		// pdfTeX warning: pdflatex (file ./Carlito-Bold.pfb): glyph `index130' undefined
		"pdfTeX warning( \\((\\d|\\w)+\\))?: |" +
		"\\* fontspec warning: |" +
		"Missing character: There is no .* in font .*!$|" +
		"A space is missing\\. (No warning)\\.)")
    private String patternWarnLatex = 
	"^(LaTeX Warning: |" +
	"LaTeX Font Warning: |" + 
	"(Package|Class) .+ Warning: |" + 
	"pdfTeX warning( \\((\\d|\\w)+\\))?: |" +
	"\\* fontspec warning: |" +
	"Missing character: There is no .* in font .*!$|" +
	"A space is missing\\. (No warning)\\.)";

    /**
     * Whether debugging of overfull/underfull hboxes/vboxes is on: 
     * If so, a bad box occurs in the last LaTeX run, a warning is displayed. 
     * For details, set $cleanUp to false, 
     * rerun LaTeX and have a look at the log-file. 
     * The default value is <code>true</code>. 
     */
    @Parameter(name = "debugBadBoxes", defaultValue = "true")
    private boolean debugBadBoxes = true;

    /**
     * Whether debugging of warnings is on: 
     * If so, a warning in the last LaTeX run is displayed. 
     * For details, set $cleanUp to false, 
     * rerun LaTeX and have a look at the log-file. 
     * The default value is <code>true</code>. 
     */
    @Parameter(name = "debugWarnings", defaultValue = "true")
    private boolean debugWarnings = true;

    /**
     * Whether creation of pdf-files from latex-files goes via dvi-files. 
     * <p>
     * If <code>pdfViaDvi</code> is set 
     * and the latex processor needs repetitions, 
     * these are all done creating dvi 
     * and then pdf is created in a final step 
     * invoking the command {@link #dvi2pdfCommand}. 
     * If <code>pdfViaDvi</code> is not set, 
     * latex is directly converted into pdf. 
     * <p>
     * Currently, not only conversion of latex-files is affected, 
     * but also conversion of graphic files 
     * into graphic formats which allow inclusion in the tex-file. 
     * If it goes via latex, 
     * then the formats are more based on (encapsulated) postscript; 
     * else on pdf. 
     * <p>
     * Of course, the target dvi is not affected: 
     * This uses always the dvi-format. 
     * What is also affected are the tasks 
     * creating html, odt or docs: 
     * Although these are based on htlatex which is always dvi-based, 
     * the preprocessing is done in dvi or in pdf. 
     * Also the task txt is affected. 
     * <p>
     * The default value is <code>false</code>. 
     */
    // if false: directly 
    @Parameter(name = "pdfViaDvi", defaultValue = "false")
    private boolean pdfViaDvi = false;

    /**
     * The driver to convert dvi into pdf-files. 
     * Note that this must fit the options 
     * of the packages <code>xcolor</code> and <code>graphicx</code>. 
     * Sensible values are 
     * <code>dvipdf</code>, <code>dvipdfm</code>, <code>dvipdfmx</code>, 
     * and <code>dvipdft</code> 
     * (which is <code>dvipdfm</code> with option <code>-t</code>). 
     * The default value is <code>dvipdfmx</code>. 
     */
    @Parameter(name = "dvi2pdfCommand", defaultValue = "dvipdfmx")
    private String dvi2pdfCommand = "dvipdfmx";

    /**
     * The options for the command {@link #dvi2pdfCommand}. 
     * The default value is the empty string. 
     */
    @Parameter(name = "dvi2pdfOptions", defaultValue = "")
    private String dvi2pdfOptions = "";

    /**
     * The pattern is applied linewise to the log-file 
     * and matching triggers rerunning {@link #latex2pdfCommand} 
     * if {@link #maxNumReRunsLatex} is not yet reached 
     * to ensure termination. 
     * <p>
     * This pattern may never be ensured to be complete, 
     * because any package 
     * may indicate the need to rerun {@link #latex2pdfCommand} 
     * with its own pattern any new package may break completeness. 
     * Nevertheless, the default value aims completeness 
     * while be tight enough not to trigger a superfluous rerun. 
     * <p>
     * If the current default value is not appropriate, 
     * please overwrite it in the configuration 
     * and notify the developer of this plugin of the deficiency. 
     * The default value is given below. 
     */
    // FIXME: default? to be replaced by an array of strings? **** 
    // FIXME: explicit tests required for each pattern. 
    // Not only those but all patterns. 
    // FIXME: seems a problem with the pattern spreading over two lines 
    @Parameter(name = "patternReRunLatex", 
	       defaultValue = 
	       // general message 
	       "^(LaTeX Warning: Label\\(s\\) may have changed\\. " 
	       + "Rerun to get cross-references right\\.$|" +
	       // default message in one line for packages 
	       "Package \\w+ Warning: .*Rerun( .*|\\.)$|" +
	       // works for 
	       // Package totcount Warning: Rerun to get correct total counts
	       // Package longtable Warning: Table widths have changed. Rerun LaTeX ...
	       // Package hyperref Warning: Rerun to get outlines right (old hyperref)
	       // Package rerunfilecheck Warning: File `...' has changed. Rerun.
	       // ... 
	       // default message in two lines for packages 
	       // FIXME: would require parsing of more than one line 
	       "Package \\w+ Warning: .*$"
	       + "^\\(\\w+\\) .*Rerun .*$|" +
	       // works for 
	       // Package natbib Warning: Citation\\(s\\) may have changed.
	       // (natbib)                Rerun to get citations correct.
	       // Package Changebar Warning: Changebar info has changed.
	       // (Changebar)                Rerun to get the bars right
	       // Package rerunfilecheck Warning: File `foo.out' has changed.
	       // (rerunfilecheck)                Rerun to get outlines right"
	       // (rerunfilecheck)                or use package `bookmark'.
	       // but not for 
	       // Package biblatex Warning: Please (re)run Biber on the file:
	       // (biblatex)                test
	       // (biblatex)                and rerun LaTeX afterwards. 
	       //
	       // messages specific to various packages 
	       "LaTeX Warning: Etaremune labels have changed\\.$|" +
	       // 'Rerun to get them right.' is on the next line
	       //
	       // from package rerunfilecheck used by other packages like new hyperref 
	       // Package rerunfilecheck Warning: File `foo.out' has changed.
	       "\\(rerunfilecheck\\)                Rerun to get outlines right$)"
	       //  (rerunfilecheck)                or use package `bookmark'.
    )
   private String patternReRunLatex = 
       // general message 
       "^(LaTeX Warning: Label\\(s\\) may have changed. " 
       + "Rerun to get cross-references right\\.$|" +
       // default message in one line for packages 
       "Package \\w+ Warning: .*Rerun .*$|" +
       // works for 
       // Package totcount Warning: Rerun to get correct total counts
       // Package longtable Warning: Table widths have changed. Rerun LaTeX ...
       // Package hyperref Warning: Rerun to get outlines right (old hyperref)
       // ... 
       // default message in two lines for packages 
       "Package \\w+ Warning: .*$"
       + "^\\(\\w+\\) .*Rerun .*$|" +
       // works for 
       // Package natbib Warning: Citation\\(s\\) may have changed.
       // (natbib)                Rerun to get citations correct.
       // Package Changebar Warning: Changebar info has changed.
       // (Changebar)                Rerun to get the bars right
       //
       // messages specific to various packages 
       "LaTeX Warning: Etaremune labels have changed\\.$|" +
       // 'Rerun to get them right.' is on the next line
       //
       // from package rerunfilecheck used by other packages like new hyperref 
       // Package rerunfilecheck Warning: File `foo.out' has changed.
       "\\(rerunfilecheck\\)                Rerun to get outlines right$)";
       // (rerunfilecheck)                or use package `xxx'.

    /**
     * The maximal allowed number of reruns of {@link #latex2pdfCommand}. 
     * This is to avoid endless repetitions. 
     * The default value is 5. 
     * This shall be non-negative 
     * or <code>-1</code> which signifies that there is no threshold. 
     */
    @Parameter(name = "maxNumReRunsLatex", defaultValue = "5")
    private int maxNumReRunsLatex = 5;


    // parameters for bibliography 


    /**
     * The BibTeX command to create a bbl-file 
     * from an aux-file and a bib-file 
     * (using a bst-style file). 
     * The default value is <code>bibtex</code>. 
     */
    @Parameter(name = "bibtexCommand", defaultValue = "bibtex")
    private String bibtexCommand = "bibtex";

    // FIXME: Any parameters for bibtex? 
// Usage: bibtex [OPTION]... AUXFILE[.aux]
//   Write bibliography for entries in AUXFILE to AUXFILE.bbl,
//   along with a log file AUXFILE.blg.
// -min-crossrefs=NUMBER  include item after NUMBER cross-refs; default 2
// -terse                 do not print progress reports
// -help                  display this help and exit
// -version               output version information and exit

// how to detect errors/warnings??? 
//Process exited with error(s)

    /**
     * The options for the command {@link #bibtexCommand}. 
     * The default value is the empty string. 
     */
    @Parameter(name = "bibtexOptions", defaultValue = "")
    private String bibtexOptions = "";



    /**
     * The Pattern in the blg-file 
     * indicating that {@link #bibtexCommand} failed. 
     * The default value is chosen 
     * according to the <code>bibtex</code> documentation. 
     */
    @Parameter(name = "patternErrBibtex", defaultValue = "error message")
    private String patternErrBibtex = "error message";

    /**
     * The Pattern in the blg-file 
     * indicating a warning {@link #bibtexCommand} emitted. 
     * The default value is chosen 
     * according to the <code>bibtex</code> documentation. 
     */
    @Parameter(name = "patternWarnBibtex", defaultValue = "^Warning--")
    private String patternWarnBibtex = "^Warning--";


    // parameters for index 


    /**
     * The MakeIndex command to create an ind-file 
     * from an idx-file logging on an ilg-file. 
     * The default value is <code>makeindex</code>. 
     */
    @Parameter(name = "makeIndexCommand", defaultValue = "makeindex")
    private String makeIndexCommand = "makeindex";

    /**
     * The options for the command {@link #makeIndexCommand}. 
     * Note that the option <code>-o xxx.ind</code> to specify the output file 
     * is not allowed because this plugin 
     * expects the output for the latex main file <code>xxx.tex</code> 
     * <code>xxx.ind</code>. 
     * Likewise, the option <code>-t xxx.ilg</code> 
     * to specify the logging file is not allowed, 
     * because this software uses the standard logging file 
     * to detect failures processing the idx-file. 
     * Also the option <code>-i</code> 
     * which specifies reading the raw index from standard input 
     * is not allowed. 
     * Specifying a style file with option <code>-s yyy.ist</code> 
     * is possible if only an index is used but no glossary. 
     * FIXME: rethink what about multiple indices. 
     * <p>
     * Note that the options specified here 
     * are also used to create glossaries. 
     * In addition for glossaries, the options 
     * <code>-s</code>, <code>-o</code> and <code>-t</code> are used. 
     * Thus also these options should not be used. 
     * The default value is the empty string. 
     * Useful options in this context are 
     * <ul>
     * <li><code>-c</code> remove blanks from index entries 
     * <li><code>-g</code> german ordering
     * <li><code>-l</code> letter ordering
     * <li><code>-r</code> without collecting index entries 
     * on 3 or more successive pages. 
     * </ul>
     */
    @Parameter(name = "makeIndexOptions", defaultValue = "")
    private String makeIndexOptions = "";

    /**
     * The Pattern in the ilg-file 
     * indicating that {@link #makeIndexCommand} failed. 
     * The default value <code>(!! Input index error )</code> 
     * is chosen according to the <code>makeindex</code> documentation. 
     */
    @Parameter(name = "patternErrMakeIndex", 
	       defaultValue = "(!! Input index error )")
    private String patternErrMakeIndex = "(!! Input index error )";

    /**
     * The Pattern in the ilg-file 
     * indicating a warning {@link #makeIndexCommand} emitted. 
     * The default value <code>(## Warning )</code> 
     * is chosen according to the <code>makeindex</code> documentation. 
     */
    @Parameter(name = "patternWarnMakeIndex", defaultValue = "(## Warning )")
    private String patternWarnMakeIndex = "(## Warning )";

    /**
     * The pattern in the log-file which triggers 
     * rerunning {@link #makeIndexCommand} 
     * followed by {@link #latex2pdfCommand}. 
     * This pattern only occurs, if package <code>rerunfilecheck</code> 
     * is used with option <code>index</code>. 
     * The default value 
     * is chosen according to the package documentation. 
     * If the user finds that default value is not appropriate, 
     * (s)he is asked to contribute 
     * and to notify the developer of this plugin. 
     */
    @Parameter(name = "patternReRunMakeIndex", 
	       defaultValue = 
	//"^Package rerunfilecheck Warning: File `.*\\.idx' has changed\\.$" //+
	"^\\(rerunfilecheck\\) +Rerun LaTeX/makeindex to get index right\\.$"
)
    // FIXME: should be included the full pattern. 
    // First part works second also but not together. 
    // Also did not find any way to connect the two parts. 
    // This gives rise to the conjecture 
    // that also other patterns do not work properly. 
    private String patternReRunMakeIndex = 
	//"^Package rerunfilecheck Warning: File `.*\\.idx' has changed\\.$" //+
	"^\\(rerunfilecheck\\) +Rerun LaTeX/makeindex to get index right\\.$";
 
    /**
     * The SplitIndex command to create ind-files 
     * from an idx-file logging on ilg-files. 
     * This command invokes {@link #makeIndexCommand}. 
     * The default value is <code>splitindex</code>. 
     */
    @Parameter(name = "splitIndexCommand", defaultValue = "splitindex")
    private String splitIndexCommand = "splitindex";

    /**
     * The options for {@link #splitIndexCommand}. 
     * Here, one has to distinguish between the options 
     * processed by {@link #splitIndexCommand} 
     * and those passed to {@link #makeIndexCommand}. 
     * The second category cannot be specified here, 
     * it is already given by {@link #makeIndexOptions}. 
     * In the first category is the option <code>-m</code> 
     * to specify the {@link #makeIndexCommand}. 
     * This is used automatically and cannot be specified here. 
     * Since {@link #splitIndexCommand} is used 
     * in conjunction with package <code>splitidx</code>, 
     * which hardcodes various parameters 
     * which are the default values for {@link #splitIndexCommand} 
     * and because the option may not alter certain interfaces, 
     * the only option which may be given explicitly 
     * is <code>-V</code>, the short cut for <code>--verbose</code>. 
     * Do not use <code>--verbose</code> either for sake of portability. 
     * The default value is <code>-V</code>; it could also be empty. 
     */
    @Parameter(name = "splitIndexOptions", defaultValue = "-V")
    private String splitIndexOptions = "splitindexOptions";


    // parameters for glossary 


    /**
     * The MakeGlossaries command to create a gls-file 
     * from a glo-file (invoked without file ending) 
     * also taking ist-file or xdy-file 
     * into account logging on a glg-file. 
     * The default value is <code>makeglossaries</code>. 
     */
    @Parameter(name = "makeGlossariesCommand", defaultValue = "makeglossaries")
    private String makeGlossariesCommand = "makeglossaries";

    /**
     * The options for the command {@link #makeGlossariesCommand}. 
     * These are the options for <code>makeindex</code> 
     * (not for {@link #makeIndexCommand}) 
     * and for <code>xindy</code> (also hardcoded). 
     * The aux-file decides on whether program is executed 
     * and consequently which options are used. 
     * <p>
     * The default value is the empty option string. 
     * Nevertheless, <code>xindy</code> is invoked as 
     * <code>xindy -L english  -I xindy -M ...</code>. 
     * With option <code>-L german</code>, this is added. 
     * Options <code>-M</code> for <code>xindy</code> 
     * <code>-s</code> for <code>makeindex</code> and 
     * <code>-t</code> and <code>-o</code> for both, 
     * <code>xindy</code> and <code>makeindex</code>. 
     */
    @Parameter(name = "makeGlossariesOptions", defaultValue = "")
    private String makeGlossariesOptions = "";

    /**
     * The Pattern in the glg-file 
     * indicating that {@link #makeGlossariesCommand} failed. 
     * The default value is <code>(^\*\*\* unable to execute: )</code>. 
     * If this is not appropriate, please modify 
     * and notify the developer of this plugin. 
     */
    @Parameter(name = "patternErrMakeGlossaries", 
	       defaultValue = "^\\*\\*\\* unable to execute: ")
    private String patternErrMakeGlossaries = 
	"^\\*\\*\\* unable to execute: ";

    /**
     * The pattern in the glg-file 
     * indicating that running <code>xindy</code> 
     * via {@link #makeGlossariesCommand} failed. 
     * The default value is <code>(^ERROR: )</code> (note the space). 
     * If this is not appropriate, please modify 
     * and notify the developer of this plugin. 
     * FIXME: This is not used. 
     */
    @Parameter(name = "patternErrXindy", defaultValue = "(^ERROR: )")
    private String patternErrXindy = "(^ERROR: )";

    /**
     * The pattern in the glg-file 
     * indicating a warning when running <code>xindy</code> 
     * via {@link #makeGlossariesCommand}. 
     * The default value is <code>(^WARNING: )</code> 
     * (note the space and the brackets). 
     * If this is not appropriate, please modify 
     * and notify the developer of this plugin. 
     */
    @Parameter(name = "patternWarnXindy", defaultValue = "(^WARNING: )")
    private String patternWarnXindy = "(^WARNING: )";

    /**
     * The pattern in the log-file which triggers 
     * rerunning {@link #makeGlossariesCommand} 
     * followed by {@link #latex2pdfCommand}. 
     * This pattern only occurs, if package <code>rerunfilecheck</code> 
     * is used with option <code>glossary</code>. 
     * The default value 
     * is chosen according to the package documentation. 
     * If the user finds that default value is not appropriate, 
     * (s)he is asked to contribute 
     * and to notify the developer of this plugin. 
     */
    @Parameter(name = "patternReRunMakeGlossaries", 
	       defaultValue = 
	//"^Package rerunfilecheck Warning: File `.*\\.glo' has changed\\.$" +
	// FIXME: really MAKEINDEX! 
	// Problem: package glossaries redefines makeglossary 
	// which breaks this solution with rerunfilecheck 
	"^\\(rerunfilecheck\\) +Rerun LaTeX/makeindex to get glossary right\\.$"
    )
    private String patternReRunMakeGlossaries = 
	//"^Package rerunfilecheck Warning: File `.*\\.glo' has changed\\.$" +
	// FIXME: really MAKEINDEX! 
	// Problem: package glossaries redefines makeglossary 
	// which breaks this solution with rerunfilecheck 
	"^\\(rerunfilecheck\\) +Rerun LaTeX/makeindex to get glossary right\\.$";


   // parameters for latex2html-conversion 

    /**
     * The tex4ht command. 
     * Possible values are e.g. 
     * <code>htlatex</code> and <code>htxelatex</code>. 
     * The default value (for which this software is also tested) 
     * is <code>htlatex</code>. 
     */
    @Parameter(name = "tex4htCommand", defaultValue = "htlatex")
    private String tex4htCommand = "htlatex";

    /**
     * The options for the <code>tex4ht</code>-style 
     * which creates a dvi-file or a pdf-file 
     * with information to create sgml, 
     * e.g. html or odt or something like that. 
     * The default value is <code>html,2</code>. 
     */
    @Parameter(name = "tex4htStyOptions", defaultValue = "html,2")
    private String tex4htStyOptions = "html,2";

    /**
     * The options for <code>tex4ht</code> which extracts information 
     * from a dvi-file or from a pdf-file 
     * into the according lg-file and idv-file producing html-files 
     * and by need and if configured accordingly 
     * svg-files, 4ct-files and 4tc-files and a css-file and a tmp-file.
     * The former two are used by <code>t4ht</code> 
     * which is configured via {@link #t4htOptions}. 
     */
    @Parameter(name = "tex4htOptions", defaultValue = "")
    private String tex4htOptions = "";

    /**
     * The options for <code>t4ht</code> which converts idv-file and lg-file 
     * into css-files, tmp-file and, 
     * by need and if configured accordingly into png files. 
     * The value <code>-p</code> prevents creation of png-pictures.
     * The default value is the empty string. 
     */
    @Parameter(name = "t4htOptions", defaultValue = "")
    private String t4htOptions = "";

    /**
     * The pattern for the target files of goal {@link Target#html} 
     * for a given latex main file <code>xxx.tex</code>. 
     * The patterns for the other targets 
     * are hardcoded and take the form 
     * <code>^T$T\.yyy$</code>, where <code>yyy</code> 
     * may be an ending or an alternative of endings. 
     * <p>
     * For an explanation of the pattern <code>T$T</code>, 
     * see {@link #patternCreatedFromLatexMain}. 
     * Spaces and newlines are removed 
     * from that pattern before processing. 
     * <p>
     * The default value has the following components: 
     * <ul>
     * <li><code>^T$T\.x?html?$</code> 
     * is the main file. 
     * <li><code>^T$Tli\d+\.x?html?$</code> 
     * are lists: toc, lof, lot, indices, glossaries, NOT the bibliography. 
     * <li><code>^T$T(ch|se|su|ap)\d+\.x?html?$</code> 
     * are chapters, sections and subsections or below 
     * and appendices. 
     * <li><code>^T$T\d+\.x?html?$</code> 
     * are footnotes. 
     * <li><code>^T$T\.css$</code> 
     * are cascaded stylesheets. 
     * <li><code>^T$T-\d+\.svg$</code> and <code>^T$T\d+x\.png$</code>
     * are svg/png-files representing figures. 
     * <li><code>^(cmsy)\d+(-c)?-\d+c?\.png$</code> 
     * represents special symbols. 
     * </ul>
     * Note that the patterns for the html-files 
     * can be summarized as <code>^T$T((ch|se|su|ap|li)?\d+)?\.x?html?$</code>. 
     * Adding the patterns for the css-file and the svg-files, we obtain 
     * <pre>
     * ^T$T(((ch|se|su|ap|li)?\d+)?\.x?html?|
     * \.css|
     * \d+x\.x?bb|
     * \d+x\.png|
     * -\d+\.svg)$
     * </pre>. 
     * <p>
     * The pattern is designed to match quite exactly 
     * the files to be copied to {@link #targetSiteDirectory}, 
     * for the goal {@link Target#html}, 
     * not much more and at any case not less. 
     * since {@link #tex4htCommand} is not well documented, 
     * and still subject to development, 
     * this pattern cannot be guaranteed to be final. 
     * If the user finds an extension, (s)he is asked to contribute 
     * and to notify the developer of this plugin. 
     * Then the default value will be extended. 
     */
    @Parameter(name = "patternT4htOutputFiles", defaultValue = "")
    private String patternT4htOutputFiles = 
	"^(T$T(((ch|se|su|ap|li)?\\d+)?\\.x?html?|" + 
	/*   */"\\.css|" + 
	/*   */"\\d+x\\.x?bb|" + 
	/*   */"\\d+x\\.png|" +
	/*   */"-\\d+\\.svg)|" + 
	"(cmsy)\\d+(-c)?-\\d+c?\\.png)$";


   // parameters for further conversions 

    /**
     * The latex2rtf command to create rtf from latex directly. 
     * The default value is <code>latex2rtf</code>. 
     */
    @Parameter(name = "latex2rtfCommand", defaultValue = "latex2rtf")
    private String latex2rtfCommand = "latex2rtf";

    /**
     * The options of the command {@link #latex2rtfCommand}. 
     * The default value is the empty string. 
     */
    @Parameter(name = "latex2rtfOptions", defaultValue = "")
    private String latex2rtfOptions = "";

    /**
     * The odt2doc command 
     * to create MS word-formats from otd-files. 
     * The default value is <code>odt2doc</code>; 
     * equivalent here is <code>unoconv</code>. 
     * Note that <code>odt2doc</code> just calls <code>unoconv</code> 
     * with odt-files as input and doc-file as default output. 
     *
     * @see #odt2docOptions
     */
    @Parameter(name = "odt2docCommand", defaultValue = "odt2doc")
    private String odt2docCommand = "odt2doc";

    /**
     * The options of the command {@link #odt2docCommand}. 
     * Above all specification of output format 
     * via the option <code>-f</code>. 
     * Invocation is <code>odt2doc -f&lt;format> &lt;file>.odt</code>. 
     * All output formats are shown by <code>odt2doc --show</code> 
     * but the formats interesting in this context 
     * are <code>doc, doc6, doc95,docbook, docx, docx7, ooxml, rtf</code>. 
     * Interesting also the verbosity options <code>-v, -vv, -vvv</code> 
     * the timeout <code>-T=secs</code> and <code>--preserve</code> 
     * to keep permissions and timestamp of the original document. 
     * The default value is <code>-fdocx</code>. 
     *
     * @see #odt2docCommand
     */
    @Parameter(name = "odt2docOptions", defaultValue = "-fdocx")
    private String odt2docOptions = "-fdocx";

    /**
     * The pdf2txt-command for converting pdf-files into plain text files. 
     * The default value is <code>pdftotext</code>. 
     *
     * @see #pdf2txtOptions
     */
    @Parameter(name = "pdf2txtCommand", defaultValue = "pdftotext")
    private String pdf2txtCommand = "pdftotext";

    /**
     * The options of the command {@link #pdf2txtCommand}. 
     * The default value is the empty string. 
     *
     * @see #pdf2txtCommand
     */
    @Parameter(name = "pdf2txtOptions", defaultValue = "-q")
    private String pdf2txtOptions = "-q";



    /**
     * The chktex-command for checking latex main files. 
     * The default value is <code>chktex</code>. 
     *
     * @see #chkTexOptions
     */
    @Parameter(name = "chkTexCommand", defaultValue = "chktex")
    private String chkTexCommand = "chktex";


    /**
     * The options of the command {@link #chkTexCommand}, 
     * except <code>-o output-file</code> 
     * specifying the output file which is added automatically. 
     * <p>
     * Here is a list of options useful in this context. 
     * The first group of these are muting options: 
     * <ul>
     * <li><code>-w</code>, <code>-e</code>, <code>-m</code>, 
     * Make the message number passed as parameter 
     * a warning/an error/a message and turns it on. 
     * Messages are not counted. 
     * <li><code>-n</code>
     * Turns the warning/error number passed as a parameter off. 
     * <li><code>-L</code>
     * Turns off suppression of messages on a per line basis. 
     * </ul>
     * The next group of interesting options are for output control: 
     * <ul>
     * <li><code>-q</code>
     * Shuts up about copyright information.
     * <li><code>-o output-file</code>
     * Specifies the output file. This is added automatically 
     * and shall thus not be specified by the user. 
     * <li><code>-b[0|1]</code>
     * If you use the -o switch, and the named outputfile exists,
     * it will be renamed to <code>filename.bak</code>.
     * <li><code>-f format</code>
     * Specifies the format of the output 
     * via a format similar to <code>printf()</code>. 
     * For details consult the manual. 
     * <li><code>-vd</code>
     * Verbosity level followed by a number <code>d</code> 
     * specifying the format of the output. 
     * The verbosity number is resolved as a pattern 
     * as if given by the option <code>-f format</code>. 
     * Thus the option <code>-v</code> is ignored 
     * if the option <code>-f format</code> is specified. 
     * </ul>
     * The default value is <code>-q -b0</code> 
     * avoiding verbose output and backing up the output log-file. 
     *
     * @see #chkTexCommand
     */
    // -v: verbosity: 
    //     - 0 File:Line:Column:Warning number:Warning message
    //         No specification on the kind of the entry 
    //     - 1 1st line: (Error|Warning|Message) in <File> line <Line>: message 
    //         2nd line: according line of the source 
    //         3rd line: cursor ^ pointing to the place where the problem is 
    //     - 2 1st line as for level 1 
    //         2nd line: line of source with pointer for the problem 
    //                   has shape: [7m [0m
    //     - 3 "File", line Line: Warning message 
    //     - 4 1st line as for 3, 
    //         2nd line as for 1 
    //         3rd line as for 1 
    // -f format: this allows to create more flexible formats as with -vxxx 
    //         to determine the kind of entry (Error|Warning|Message) 
    //         if kind is given, it must be at the beginning of the line 
    // -q: no copyright information 
    // -b: toggle creation of backup file: with -o: yes, additional -b: no 
    //     explicitly as -b0 and -b1, respectively. 
    @Parameter(name = "chkTexOptions", defaultValue = "")
    private String chkTexOptions = "-q -b0";

    // getter methods partially implementing default values. 


    // private File getBaseDirectory() throws BuildFailureException {
    // 	if (!(this.baseDirectory.exists() && 
    // 	      this.baseDirectory.isDirectory())) {
    // 	    throw new BuildFailureException
    // 		("The base directory '" + this.baseDirectory + 
    // 		 "' should be an existing directory, but is not. ");
    // 	}
    // 	return this.baseDirectory;
    // }

    // private File getTargetDirectory() {
    //     return this.targetDirectory;
    // }

    // private File getTargetSiteDirectory() {
    //     return this.targetSiteDirectory;
    // }

    /**
     *
     * @throws BuildFailureException 
     *    TSS01 if the tex source directory does either not exist 
     *    or is not a directory. 
     */
    // used in LatexProcessor only: 
    // .create() to determine the output directory of the created files 
    public File getTexSrcDirectoryFile() throws BuildFailureException {
	if (!(this.texSrcDirectoryFile.exists() && 
	      this.texSrcDirectoryFile.isDirectory())) {
	    throw new BuildFailureException
		("TSS01: The tex source directory '" + 
		 this.texSrcDirectoryFile + 
		 "' should be an existing directory, but is not. ");
	}
	return this.texSrcDirectoryFile;
    }

    /**
     *
     * @throws BuildFailureException 
     *    TSS02 if the tex source processing directory does either not exist 
     *    or is not a directory. 
     */
    // used in LatexProcessor only: 
    // .create() to determine which directories to be processed 
    // .processGraphics() to get all graphics files 
    // .clearAll() 
    public File getTexSrcProcDirectoryFile() throws BuildFailureException {
	if (!(this.texSrcProcDirectoryFile.exists() && 
	      this.texSrcProcDirectoryFile.isDirectory())) {
	    throw new BuildFailureException 
		("TSS02: The tex source processing directory '" + 
		 this.texSrcProcDirectoryFile + 
		 "' should be an existing directory, but is not. ");
	}
	
	return this.texSrcProcDirectoryFile;
    }

    public boolean getReadTexSrcProcDirRec() {
	return   this.readTexSrcProcDirRec;
    }

    /**
     *
     * @throws BuildFailureException 
     *    TSS03 if the output directory exists and is no directory. 
     */
    public File   getOutputDirectoryFile() throws BuildFailureException {
	if (/**/this.outputDirectoryFile.exists() && 
	    !   this.outputDirectoryFile.isDirectory()) {
	    throw new BuildFailureException
		("TSS03: The output directory '" + this.outputDirectoryFile + 
		 "' should be a directory if it exists, but is not. ");
	}
       return this.outputDirectoryFile;
    }

    public SortedSet<Target> getTargetSet() {
	String[] targetSeq = this.targets.split(" *, *");
	// TreeSet is sorted. maybe this determines ordering of targets. 
	SortedSet<Target> targetSet = new TreeSet<Target>();
	for (int idx = 0; idx < targetSeq.length; idx++) {
	    targetSet.add(Target.valueOf(targetSeq[idx]));
	}
	return targetSet;
    }

    public String getPatternLatexMainFile() {
	return  this.patternLatexMainFile;
    }


    // texPath, commands and arguments 

    public File  getTexPath() {
        return this.texPath;
    }

    public boolean isCleanUp() {
        return  this.cleanUp;
    }

    public String getPatternCreatedFromLatexMain() {
	return  this.patternCreatedFromLatexMain;
    }


    public String getFig2devCommand() {
        return  this.fig2devCommand;
    }

    public String getFig2devGenOptions() {
        return  this.fig2devGenOptions;
    }

    public String getFig2devPtxOptions() {
        return  this.fig2devPtxOptions;
    }

    public String getFig2devPdfEpsOptions() {
        return  this.fig2devPdfEpsOptions;
    }

    public String getGnuplotCommand() {
        return  this.gnuplotCommand;
    }

    public String getGnuplotOptions() {
        return  this.gnuplotOptions;
    }

    public String getMetapostCommand() {
        return  this.metapostCommand;
    }

    public String getMetapostOptions() {
        return  this.metapostOptions;
    }

    public String getSvg2devCommand() {
	return  this.svg2devCommand;
    }

    public String getSvg2devOptions() {
	return  this.svg2devOptions;
    }

    public String getEbbCommand() {
        return  this.ebbCommand;
    }
    public String getEbbOptions() {
        return  this.ebbOptions;
    }


// FIXME: to be renamed: latex2pdf-command 
    public String getLatex2pdfCommand() {
        return  this.latex2pdfCommand;
    }

    // FIXME: to be renamed: texOptions 
    public String getLatex2pdfOptions() {
        return  this.latex2pdfOptions;
    }

    public String getPatternErrLatex() {
	return  this.patternErrLatex;
    }

    // same pattern as for latex 
    public String getPatternErrMPost() {
	return  this.patternErrLatex;
    }

    public String getPatternWarnLatex() {
	return  this.patternWarnLatex;
    }

    public boolean getDebugBadBoxes() {
	return   this.debugBadBoxes;
    }

    public boolean getDebugWarnings() {
	return   this.debugWarnings;
    }

    public     LatexDev getPdfViaDvi() {
	return LatexDev.   devViaDvi(this.pdfViaDvi);
    }

    public String getDvi2pdfCommand() {
	return  this.dvi2pdfCommand;
    }

    public String getDvi2pdfOptions() {
	return  this.dvi2pdfOptions;
    }

    public String getPatternReRunLatex() {
	return  this.patternReRunLatex;
    }

    public int   getMaxNumReRunsLatex() {
	return this.maxNumReRunsLatex;
    }


    public String getBibtexCommand() {
        return  this.bibtexCommand;
    }
    public String getBibtexOptions() {
        return  this.bibtexOptions;
    }

    public String getPatternErrBibtex() {
	return  this.patternErrBibtex;
    }

    public String getPatternWarnBibtex() {
	return  this.patternWarnBibtex;
    }


    public String getMakeIndexCommand() {
	return  this.makeIndexCommand;
    }

    public String getMakeIndexOptions() {
	return  this.makeIndexOptions;
    }

    public String getPatternErrMakeIndex() {
	return  this.patternErrMakeIndex;
    }

    public String getPatternWarnMakeIndex() {
	return this.patternWarnMakeIndex;
    }

    public String getPatternReRunMakeIndex() {
	return this.patternReRunMakeIndex;
    }

    public String getSplitIndexCommand() {
	return  this.splitIndexCommand;
    }

    public String getSplitIndexOptions() {
	return  this.splitIndexOptions;
    }

    public String getMakeGlossariesCommand() {
	return  this.makeGlossariesCommand;
    }

    public String getMakeGlossariesOptions() {
	return  this.makeGlossariesOptions;
    }

    public String getPatternErrMakeGlossaries() {
	return  this.patternErrMakeGlossaries;
    }

    public String getPatternWarnXindy() {
	return  this.patternWarnXindy;
    }

    public String getPatternReRunMakeGlossaries() {
	return  this.patternReRunMakeGlossaries;
    }


    public String getTex4htCommand() {
        return  this.tex4htCommand;
    }

    public String getTex4htStyOptions() {
        return  this.tex4htStyOptions;
    }

    public String getTex4htOptions() {
        return  this.tex4htOptions;
    }

    public String getT4htOptions() {
        return  this.t4htOptions;
    }

    public String getPatternT4htOutputFiles() {
	return  this.patternT4htOutputFiles;
    }

    public String getLatex2rtfCommand() {
        return  this.latex2rtfCommand;
    }

    public String getLatex2rtfOptions() {
        return  this.latex2rtfOptions;
    }

    public String getOdt2docCommand() {
        return  this.odt2docCommand;
    }

    public String getOdt2docOptions() {
	return  this.odt2docOptions;
    }

    public String getPdf2txtCommand() {
        return  this.pdf2txtCommand;
    }

    public String getPdf2txtOptions() {
        return  this.pdf2txtOptions;
    }

    public String getChkTexCommand() {
        return  this.chkTexCommand;
    }

    public String getChkTexOptions() {
        return  this.chkTexOptions;
    }



    // setter methods 

   /**
     * Sets {@link #baseDirectory} and updates 
     * {@link #texSrcDirectoryFile} and {@link #texSrcProcDirectoryFile}. 
     */
    public void setBaseDirectory(File baseDirectory) {
        this.baseDirectory = baseDirectory;
	this.texSrcDirectoryFile     = new File(this.baseDirectory, 
						this.texSrcDirectory);
	this.texSrcProcDirectoryFile = new File(this.texSrcDirectoryFile,
						this.texSrcProcDirectory);	
    }

    /**
     * Sets {@link #targetDirectory}. 
     */
    public void setTargetDirectory(File targetDirectory) {
        this.targetDirectory = targetDirectory;
    }

    /**
     * Sets {@link #targetSiteDirectory} and updates 
     * {@link #outputDirectoryFile}. 
     */
    public void setTargetSiteDirectory(File targetSiteDirectory) {
        this.targetSiteDirectory = targetSiteDirectory;
	this.outputDirectoryFile = new File(this.targetSiteDirectory, 
					    this.outputDirectory);
    }

    /**
     * Sets {@link #texSrcDirectory} and updates 
     * {@link #texSrcDirectoryFile} and {@link #texSrcProcDirectoryFile}. 
     */
    public void setTexSrcDirectory(String texSrcDirectory) {
        this.texSrcDirectory = texSrcDirectory;
	this.texSrcDirectoryFile     = new File(this.baseDirectory, 
						this.texSrcDirectory);
	this.texSrcProcDirectoryFile = new File(this.texSrcDirectoryFile,
						this.texSrcProcDirectory);
    }

    /**
     * Sets {@link #texSrcProcDirectory} and updates 
     * {@link #texSrcProcDirectoryFile}. 
     */
    public void setTexSrcProcDirectory(String texSrcProcDirectory) {
	this.texSrcProcDirectory = texSrcProcDirectory;
	this.texSrcProcDirectoryFile = new File(this.texSrcDirectoryFile,
						this.texSrcProcDirectory);
    }

    public void setReadTexSrcProcDirRec(boolean readTexSrcDirProcRec) {
	this.readTexSrcProcDirRec = readTexSrcProcDirRec;
    }

    /**
     * Sets {@link #outputDirectory} and updates {@link #outputDirectoryFile}. 
     */
    public void setOutputDirectory(String outputDirectory) {
        this.outputDirectory = outputDirectory;
	this.outputDirectoryFile = new File(this.targetSiteDirectory, 
					    this.outputDirectory);
    }

    public void setTargets(String targets) {
	this.targets = targets.trim();
    }

    // setter method for patternLatexMainFile in maven 
    // trims parameter before setting 
    public void setPatternLatexMainFile(String patternLatexMainFile) {
	this.patternLatexMainFile = patternLatexMainFile
	    .replaceAll("(\t|\n)+", "").trim();
    }

    // method introduces patternLatexMainFile in ant 
    public PatternLatexMainFile createPatternLatexMainFile() {
   	return new PatternLatexMainFile();
    }

    // defines patternLatexMainFile element with text in ant 
    public class PatternLatexMainFile {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternLatexMainFile(pattern);
   	}
    }

    public void setTexPath(File texPath) {
        this.texPath = texPath;
    }

    public void setCleanUp(boolean cleanUp) {
        this.cleanUp = cleanUp;
    }

    // FIXME: as patternCreatedFromLatexMain 
    // replace "\n" (canonical newline in xml) also for other patterns by ""

    // setter method for patternCreatedFromLatexMain in maven 
    // eliminates tab, newline and blanks and trims parameter before setting 
    public void setPatternCreatedFromLatexMain(String pattern) {
	this.patternCreatedFromLatexMain = pattern
	    .replaceAll("(\t|\n| )+", "").trim();
    }

    // method introduces patternCreatedFromLatexMain in ant 
    public PatternCreatedFromLatexMain createPatternCreatedFromLatexMain() {
   	return new PatternCreatedFromLatexMain();
    }

    // defines patternCreatedFromLatexMain element with text in ant 
    public class PatternCreatedFromLatexMain {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternCreatedFromLatexMain(pattern);
   	}
    }

    public void setFig2devCommand(String fig2devCommand) {
        this.fig2devCommand = fig2devCommand;
    }

    public void setFig2devGenOptions(String fig2devGenOptions) {
        this.fig2devGenOptions = fig2devGenOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setFig2devPtxOptions(String fig2devPtxOptions) {
        this.fig2devPtxOptions = fig2devPtxOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setFig2devPdfEpsOptions(String fig2devPdfEpsOptions) {
        this.fig2devPdfEpsOptions = fig2devPdfEpsOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setGnuplotCommand(String gnuplotCommand) {
        this.gnuplotCommand = gnuplotCommand;
    }

    public void setGnuplotOptions(String gnuplotOptions) {
        this.gnuplotOptions = gnuplotOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setMetapostCommand(String metapostCommand) {
        this.metapostCommand = metapostCommand;
    }

    // setter method for metapostOptions in maven 
    public void setMetapostOptions(String metapostOptions) {
        this.metapostOptions = metapostOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    // method introduces metapostOptions in ant 
    public MetapostOptions createMetapostOptions() {
   	return new MetapostOptions();
    }

   // defines e element with text in ant 
    public class MetapostOptions {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String args) {
   	    Settings.this.setMetapostOptions(args);
   	}
    }

    public void setSvg2devCommand(String svg2devCommand) {
	this.svg2devCommand = svg2devCommand;
    }

    public void setSvg2devOptions(String svg2dev) {
	this.svg2devOptions = svg2devOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setEbbCommand(String ebbCommand) {
        this.ebbCommand = ebbCommand;
    }

    public void setEbbOptions(String ebbOptions) {
        this.ebbOptions = ebbOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setLatex2pdfCommand(String latex2pdfCommand) {
        this.latex2pdfCommand = latex2pdfCommand;
    }

    /**
     * Sets the argument string of the latex command 
     * given by {@link #latex2pdfCommand}. 
     * It is ensured that {@link #latex2pdfOptions} 
     * consist of proper options separated by a single blank. 
     *
     * @param args
     *    The arguments string to use when calling LaTeX 
     *    via {@link #latex2pdfCommand}. 
     *    Leading and trailing blank and newline are ignored. 
     *    Proper arguments are separated by blank and newline. 
     */
    // setter method for latex2pdfOptions in maven 
    public void setLatex2pdfOptions(String args) {
        this.latex2pdfOptions = args.replaceAll("(\t|\n| )+", " ").trim();
    }

    // method introduces latex2pdfOptions in ant 
    public Latex2pdfOptions createLatex2pdfOptions() {
   	return new Latex2pdfOptions();
    }

   // defines e element with text in ant 
    public class Latex2pdfOptions {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String args) {
   	    Settings.this.setLatex2pdfOptions(args);
   	}
    }

 
    // setter method for patternErrLatex in maven 
    public void setPatternErrLatex(String patternErrLatex) {
	this.patternErrLatex = patternErrLatex;
    }

    // method introduces patternErrLatex in ant 
    public PatternErrLatex createPatternErrLatex() {
   	return new PatternErrLatex();
    }

    // defines patternErrLatex element with text in ant 
    public class PatternErrLatex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternErrLatex(pattern);
   	}
    }



    // setter method for patternWarnLatex in maven 
    public void setPatternWarnLatex(String patternWarnLatex) {
	this.patternWarnLatex = patternWarnLatex
	    .replaceAll("(\t|\n)+", "").trim();
    }

    // method introduces patternWarnLatex in ant 
    public PatternWarnLatex createPatternWarnLatex() {
   	return new PatternWarnLatex();
    }

    // defines patternWarnLatex element with text in ant 
    public class PatternWarnLatex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternWarnLatex(pattern);
   	}
    }

    public void setDebugBadBoxes(boolean debugBadBoxes) {
	this.debugBadBoxes = debugBadBoxes;
    }

    public void setDebugWarnings(boolean debugWarnings) {
	this.debugWarnings = debugWarnings;
    }

    public void setPdfViaDvi(boolean pdfViaDvi) {
	this.pdfViaDvi = pdfViaDvi;
    }

    public void setDvi2pdfCommand(String dvi2pdfCommand) {
	this.dvi2pdfCommand = dvi2pdfCommand;
    }

    public void setDvi2pdfOptions(String dvi2pdfOptions) {
	this.dvi2pdfOptions = dvi2pdfOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    // setter method for patternReRunLatex in maven 
    public void setPatternReRunLatex(String pattern) {
	this.patternReRunLatex = pattern
	    .replaceAll("(\t|\n)+", "").trim();
    }

    // method introduces patternReRunLatex in ant 
    public PatternReRunLatex createPatternReRunLatex() {
   	return new PatternReRunLatex();
    }

    // defines patternNeedAnotherLatexRun element with text in ant 
    public class PatternReRunLatex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternReRunLatex(pattern);
   	}
    }

    // FIXME: real check needed. also in other locations. 
    public void setMaxNumReRunsLatex(int maxNumReRunsLatex) {
	assert maxNumReRunsLatex >= 1 || maxNumReRunsLatex == -1
	    : "Found illegal max number of reruns "+maxNumReRunsLatex+". ";
	this.maxNumReRunsLatex = maxNumReRunsLatex;
    }


    public void setBibtexCommand(String bibtexCommand) {
        this.bibtexCommand = bibtexCommand;
    }
    public void setBibtexOptions(String bibtexOptions) {
        this.bibtexOptions = bibtexOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    // setter method for patternErrBibtex in maven 
    public void setPatternErrBibtex(String patternErrBibtex) {
        this.patternErrBibtex = patternErrBibtex;
    }

    // method introduces patternErrBibtex in ant 
    public PatternErrBibtex createPatternErrBibtex() {
   	return new PatternErrBibtex();
    }

    // defines patternErrBibtex element with text in ant 
    public class PatternErrBibtex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternErrBibtex(pattern);
   	}
    }

    // setter method for patternWarnBibtex in maven 
    public void setPatternWarnBibtex(String patternWarnBibtex) {
        this.patternWarnBibtex = patternWarnBibtex;
    }

    // method introduces patternWarnBibtex in ant 
    public PatternWarnBibtex createPatternWarnBibtex() {
   	return new PatternWarnBibtex();
    }

    // defines patternWarnBibtex element with text in ant 
    public class PatternWarnBibtex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternWarnBibtex(pattern);
   	}
    }


    public void setMakeIndexCommand(String makeIndexCommand) {
        this.makeIndexCommand = makeIndexCommand;
    }

    public void setMakeIndexOptions(String makeIndexOptions) {
	this.makeIndexOptions = makeIndexOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    // setter method for patternErrMakeIndex in maven 
    public void setPatternErrMakeIndex(String patternErrMakeIndex) {
        this.patternErrMakeIndex = patternErrMakeIndex
	    .replaceAll("\n+", "").trim();
    }

    // method introduces patternErrMakeIndex in ant 
    public PatternErrMakeIndex createPatternErrMakeIndex() {
   	return new PatternErrMakeIndex();
    }

    // defines patternErrMakeIndex element with text in ant 
    public class PatternErrMakeIndex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternErrMakeIndex(pattern);
   	}
    }

    // FIXME: MakeIndex
    // setter method for patternWarnMakeIndex in maven 
    public void setPatternWarnMakeIndex(String patternWarnMakeIndex) {
        this.patternWarnMakeIndex = patternWarnMakeIndex
	    .replaceAll("\n+", "").trim();
    }

    // method introduces patternWarnMakeIndex in ant 
    public PatternWarnMakeIndex createPatternWarnMakeIndex() {
   	return new PatternWarnMakeIndex();
    }

    // defines patternWarnMakeIndex element with text in ant 
    public class PatternWarnMakeIndex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternWarnMakeIndex(pattern);
   	}
    }

    // setter method for patternReRunMakeIndex in maven 
    public void setPatternReRunMakeIndex(String pattern) {
        this.patternReRunMakeIndex = pattern
	    .replaceAll("\n+", "").trim();
    }

    // method introduces patternMakeIndex in ant 
    public PatternReRunMakeIndex createPatternReRunMakeIndex() {
   	return new PatternReRunMakeIndex();
    }

    // defines patternReRunMakeIndex element with text in ant 
    public class PatternReRunMakeIndex {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternReRunMakeIndex(pattern);
   	}
    }

    public void setSplitIndexCommand(String splitIndexCommand) {
        this.splitIndexCommand = splitIndexCommand;
    }

    public void setSplitIndexOptions(String splitIndexOptions) {
        this.splitIndexOptions = splitIndexOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setMakeGlossariesCommand(String makeGlossariesCommand) {
        this.makeGlossariesCommand = makeGlossariesCommand;
    }

    public void setMakeGlossariesOptions(String makeGlossariesOptions) {
	this.makeGlossariesOptions = makeGlossariesOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setPatternErrMakeGlossaries(String patternErrMakeGlossaries) {
	this      .patternErrMakeGlossaries =      patternErrMakeGlossaries
	    .replaceAll("\n+", "").trim();
    }

     public void setPatternWarnXindy(String patternWarnXindy) {
	this.patternWarnXindy = patternWarnXindy
	    .replaceAll("\n+", "").trim();
    }

    // setter method for patternReRunMakeGlossaries in maven 
    public void setPatternReRunMakeGlossaries(String pattern) {
        this.patternReRunMakeGlossaries = pattern
	    .replaceAll("\n+", "").trim();
    }

    // method introduces patternReRunMakeGlossaries in ant 
    public PatternReRunMakeGlossaries createPatternReRunMakeGlossaries() {
   	return new PatternReRunMakeGlossaries();
    }

    // defines patternReRunMakeGlossaries element with text in ant 
    public class PatternReRunMakeGlossaries {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternReRunMakeGlossaries(pattern);
   	}
    }

 
    public void setTex4htCommand(String tex4htCommand) {
        this.tex4htCommand = tex4htCommand;
    }

    public void setTex4htStyOptions(String tex4htStyOptions) {
	this.tex4htStyOptions = tex4htStyOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
   }

    public void setTex4htOptions(String tex4htOptions) {
	this.tex4htOptions = tex4htOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

     public void setT4htOptions(String t4htOptions) {
	this.t4htOptions = t4htOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    // setter method for patternT4htOutputFiles in maven 
    public void setPatternT4htOutputFiles(String patternT4htOutputFiles) {
	this.patternT4htOutputFiles = patternT4htOutputFiles
	    .replaceAll("(\t|\n| )+", "").trim();
    }

    // method introduces patternT4htOutputFiles in ant 
    public PatternT4htOutputFiles createPatternT4htOutputFiles() {
   	return new PatternT4htOutputFiles();
    }

    // defines patternT4htOutputFiles element with text in ant 
    public class PatternT4htOutputFiles {
	// FIXME: this is without property resolution. 
	// to add this need  pattern = getProject().replaceProperties(pattern)
	// with Task.getProject() 
   	public void addText(String pattern) {
   	    Settings.this.setPatternT4htOutputFiles(pattern);
   	}
    }

    public void setLatex2rtfCommand(String latex2rtfCommand) {
        this.latex2rtfCommand = latex2rtfCommand;
    }

    // FIXME: replaceAll: should be unified. 
    public void setLatex2rtfOptions(String latex2rtfOptions) {
        this.latex2rtfOptions = latex2rtfOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setOdt2docCommand(String odt2docCommand) {
        this.odt2docCommand = odt2docCommand;
     }

    public void setOdt2docOptions(String odt2docOptions) {
        this.odt2docOptions = odt2docOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
     }

    public void setPdf2txtCommand(String pdf2txtCommand) {
        this.pdf2txtCommand = pdf2txtCommand;
    }

    public void setPdf2txtOptions(String pdf2txtOptions) {
        this.pdf2txtOptions = pdf2txtOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }

    public void setChkTexCommand(String chkTexCommand) {
        this.chkTexCommand = chkTexCommand;
    }

    public void setChkTexOptions(String chkTexOptions) {
        this.chkTexOptions = chkTexOptions
	    .replaceAll("(\t|\n| )+", " ").trim();
    }


    public String toString() {
        StringBuilder sb = new StringBuilder();
	// directories and related 
	sb.append("[ baseDirectory=")       .append(this.baseDirectory);
	sb.append(", targetDirectory=")     .append(this.targetDirectory);
	sb.append(", targetSiteDirectory=") .append(this.targetSiteDirectory);
	sb.append(", texSrcDirectory=")     .append(this.texSrcDirectory);
	sb.append(", texSrcProcDirectory=") .append(this.texSrcProcDirectory);
	sb.append(", readTexSrcProcDirRec=").append(this.readTexSrcProcDirRec);
 	sb.append(", outputDirectory=")     .append(this.outputDirectory);
	// general parameters
 	sb.append(", targets=")             .append(this.targets);
 	sb.append(", patternLatexMainFile=").append(this.patternLatexMainFile);
        sb.append(", texPath=")             .append(this.texPath);
	sb.append(", cleanUp=")             .append(this.cleanUp);
	sb.append(", patternCreatedFromLatexMain=")
	    .append(this.patternCreatedFromLatexMain);

	// parameters for graphical preprocessors 
        sb.append(", fig2devCommand=")      .append(this.fig2devCommand);
        sb.append(", fig2devGenOptions")    .append(this.fig2devGenOptions);
        sb.append(", fig2devPtxOptions")    .append(this.fig2devPtxOptions);
        sb.append(", fig2devPdfEpsOptions") .append(this.fig2devPdfEpsOptions);
        sb.append(", gnuplotCommand=")      .append(this.gnuplotCommand);
        sb.append(", gnuplotOptions=")      .append(this.gnuplotOptions);
        sb.append(", metapostCommand=")     .append(this.metapostCommand);
        sb.append(", metapostOptions=")     .append(this.metapostOptions);
	sb.append(", svg2devCommand=")      .append(this.svg2devCommand);
	sb.append(", svg2devOptions=")      .append(this.svg2devOptions);
  
	sb.append(", ebbCommand=")          .append(this.ebbCommand);
        sb.append(", ebbOptions=")          .append(this.ebbOptions);

	// parameters for latex2pdf
        sb.append(", latex2pdfCommand=")    .append(this.latex2pdfCommand);
	sb.append(", latex2pdfOptions=")    .append(this.latex2pdfOptions);
	sb.append(", patternErrLatex=")     .append(this.patternErrLatex);
	sb.append(", patternWarnLatex=")    .append(this.patternWarnLatex);
 	sb.append(", debugBadBoxes=")       .append(this.debugBadBoxes);
 	sb.append(", debugWarnings=")       .append(this.debugWarnings);
 	sb.append(", pdfViaDvi=")           .append(this.pdfViaDvi);
 	sb.append(", dvi2pdfCommand=")      .append(this.dvi2pdfCommand);
 	sb.append(", dvi2pdfOptions=")      .append(this.dvi2pdfOptions);

	sb.append(", patternReRunLatex=")   .append(this.patternReRunLatex);
	sb.append(", maxNumReRunsLatex=")   .append(this.maxNumReRunsLatex);
	// parameters for BibTeX
        sb.append(", bibtexCommand=")       .append(this.bibtexCommand);
        sb.append(", bibtexOptions=")       .append(this.bibtexOptions);
        sb.append(", patternErrBibtex=")    .append(this.patternErrBibtex);
        sb.append(", patternWarnBibtex=")   .append(this.patternWarnBibtex);
	// parameters for MakeIndex 
        sb.append(", makeIndexCommand=")    .append(this.makeIndexCommand);
        sb.append(", makeIndexOptions=")    .append(this.makeIndexOptions);
        sb.append(", patternErrMakeIndex=") .append(this.patternErrMakeIndex);
        sb.append(", patternWarnMakeIndex=").append(this.patternWarnMakeIndex);
        sb.append(", patternReRunMakeIndex=")
	    .append(this.patternReRunMakeIndex);
	sb.append(", splitIndexCommand=")    .append(this.splitIndexCommand);
	sb.append(", splitIndexOptions=")    .append(this.splitIndexOptions);
 	// parameters for MakeGlossaries
        sb.append(", makeGlossariesCommand=")
	    .append(this.makeGlossariesCommand);
        sb.append(", makeGlossariesOptions=")
	    .append(this.makeGlossariesOptions);
	sb.append(", patternErrMakeGlossaries=")
	   .append(this.patternErrMakeGlossaries);
	sb.append(", patternWarnXindy=").append(this.patternWarnXindy);
        sb.append(", patternReRunMakeGlossaries=")
	    .append(this.patternReRunMakeGlossaries);
	// parameters for latex2html 
        sb.append(", tex4htCommand=")       .append(this.tex4htCommand);
        sb.append(", tex4htStyOptions=")    .append(this.tex4htStyOptions);
        sb.append(", tex4htOptions=")       .append(this.tex4htOptions);
	sb.append(", t4htOptions=")         .append(this.t4htOptions);
	sb.append(", patternT4htOutputFiles=")
	    .append(this.patternT4htOutputFiles);
	// parameters for latex2rtf 
        sb.append(", latex2rtfCommand=")    .append(this.latex2rtfCommand);
        sb.append(", latex2rtfOptions=")    .append(this.latex2rtfOptions);
	// parameters for odt2doc
        sb.append(", odt2docCommand=")      .append(this.odt2docCommand);
        sb.append(", odt2docOptions=")      .append(this.odt2docOptions);
	// parameters for pdf2txt 
        sb.append(", pdf2txtCommand=")      .append(this.pdf2txtCommand);
        sb.append(", pdf2txtOptions=")      .append(this.pdf2txtOptions);
	// parameters for chktex
	sb.append(", chkTexCommand=")      .append(this.chkTexCommand);
        sb.append(", chkTexOptions=")      .append(this.chkTexOptions);

	sb.append(']');
        return sb.toString();
    }

    public static void main(String[] args) {
	System.out.println("texpath: "+new Settings().getTexPath());
    }
}
